# tools/kernel_bootstrap.py
import numpy as np
from dataclasses import dataclass
from typing import Callable, Dict, Any, Tuple

@dataclass
class BootSpec:
    reps: int = 400     # adjust in YAML if desired
    block: int = 0      # 0 = auto
    seed: int = 1337

def _auto_block(n: int) -> int:
    if n <= 64:  return max(2, n // 4)
    if n <= 256: return max(4, n // 8)
    return max(8, min(n // 32, n // 3))

def _block_indices(n: int, blk: int, rng: np.random.Generator) -> np.ndarray:
    nb = max(1, n // blk)
    starts = rng.integers(0, n - blk + 1, size=nb, dtype=np.int64)
    idx = np.concatenate([np.arange(int(s), int(s)+blk) for s in starts])
    if idx.size > n: idx = idx[:n]
    return idx

def bootstrap_over_kernel(
    kernel: np.ndarray,
    compute_scalar: Callable[[np.ndarray, Dict[str, Any]], float],
    compute_kwargs: Dict[str, Any],
    spec: BootSpec
) -> Tuple[float, float, Dict[str, Any]]:
    x = np.asarray(kernel, dtype=float).ravel()
    n = x.size
    blk = spec.block or _auto_block(n)
    rng = np.random.default_rng(spec.seed)

    mu0 = float(compute_scalar(x, compute_kwargs))

    samples = np.empty(spec.reps, dtype=float)
    for i in range(spec.reps):
        idx = _block_indices(n, blk, rng)
        xb = x[idx]
        samples[i] = float(compute_scalar(xb, compute_kwargs))

    se = float(samples.std(ddof=1))
    meta = {"strategy":"kernel_block_bootstrap","reps":spec.reps,"block":blk,"n":int(n)}
    return mu0, se, meta
